/* Reverse Engineer's Hex Editor
 * Copyright (C) 2022 Daniel Collins <solemnwarning@solemnwarning.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "../src/platform.hpp"
#include <gtest/gtest.h>
#include <stdexcept>
#include <stdio.h>
#include <string.h>

#include "../src/IntelHexExport.hpp"
#include "../src/SharedDocumentPointer.hpp"

using namespace REHex;

class TempFilename
{
	public:
		char tmpfile[L_tmpnam];
		
		TempFilename()
		{
			if(tmpnam(tmpfile) == NULL)
			{
				throw std::runtime_error("Cannot generate temporary file name");
			}
			
#ifdef _WIN32
			/* > Note than when a file name is pre-pended with a backslash and no path
			 * > information, such as \fname21, this indicates that the name is valid
			 * > for the current working directory.
			 * - MSDN
			 *
			 * Sure, that makes total sense.
			*/
			if(tmpfile[0] == '\\' && strchr((tmpfile + 1), '\\') == NULL)
			{
				/* Remove the leading slash. */
				memmove(tmpfile, tmpfile + 1, strlen(tmpfile) - 1);
			}
#endif
		}
		
		~TempFilename()
		{
			unlink(tmpfile);
		}
};

#define EXPECT_FILE(t) \
{ \
	std::string content; \
	FILE *fh = fopen(tmp.tmpfile, "rb"); \
	if(fh == NULL) \
	{ \
		throw std::runtime_error(strerror(errno)); \
	} \
	char buf[1024]; \
	int l; \
	while((l = fread(buf, 1, sizeof(buf), fh)) > 0) \
	{ \
		content.append(buf, l); \
	} \
	fclose(fh); \
	EXPECT_EQ(content, t); \
}

TEST(IntelHexExport, ExportsPlainFile)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, false, IntelHexAddressingMode::IHA_16BIT, NULL, NULL);
	
	EXPECT_FILE(
		":10000000000102030405060708090A0B0C0D0E0F78\n"
		":10001000101112131415161718191A1B1C1D1E1F68\n"
		":10002000202122232425262728292A2B2C2D2E2F58\n"
		":10003000303132333435363738393A3B3C3D3E3F48\n"
		":10004000404142434445464748494A4B4C4D4E4F38\n"
		":10005000505152535455565758595A5B5C5D5E5F28\n"
		":10006000606162636465666768696A6B6C6D6E6F18\n"
		":10007000707172737475767778797A7B7C7D7E7F08\n"
		":10008000808182838485868788898A8B8C8D8E8FF8\n"
		":10009000909192939495969798999A9B9C9D9E9FE8\n"
		":1000A000A0A1A2A3A4A5A6A7A8A9AAABACADAEAFD8\n"
		":1000B000B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC8\n"
		":1000C000C0C1C2C3C4C5C6C7C8C9CACBCCCDCECFB8\n"
		":1000D000D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFA8\n"
		":1000E000E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEF98\n"
		":1000F000F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF88\n"
		":00000001FF\n");
}

TEST(IntelHexExport, ExportsFileWithVirtualSegments)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(0,     0,  64);
	doc->set_virt_mapping(64,  128, 128);
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_16BIT, NULL, NULL);
	
	EXPECT_FILE(
		":10000000000102030405060708090A0B0C0D0E0F78\n"
		":10001000101112131415161718191A1B1C1D1E1F68\n"
		":10002000202122232425262728292A2B2C2D2E2F58\n"
		":10003000303132333435363738393A3B3C3D3E3F48\n"
		":10008000808182838485868788898A8B8C8D8E8FF8\n"
		":10009000909192939495969798999A9B9C9D9E9FE8\n"
		":1000A000A0A1A2A3A4A5A6A7A8A9AAABACADAEAFD8\n"
		":1000B000B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC8\n"
		":1000C000C0C1C2C3C4C5C6C7C8C9CACBCCCDCECFB8\n"
		":1000D000D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFA8\n"
		":1000E000E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEF98\n"
		":1000F000F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF88\n"
		":00000001FF\n");
}


TEST(IntelHexExport, ExportsPlainFileWithVirtualSegments)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(0,     0, 64);
	doc->set_virt_mapping(64,  128, 64);
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, false, IntelHexAddressingMode::IHA_16BIT, NULL, NULL);
	
	EXPECT_FILE(
		":10000000000102030405060708090A0B0C0D0E0F78\n"
		":10001000101112131415161718191A1B1C1D1E1F68\n"
		":10002000202122232425262728292A2B2C2D2E2F58\n"
		":10003000303132333435363738393A3B3C3D3E3F48\n"
		":10004000808182838485868788898A8B8C8D8E8F38\n"
		":10005000909192939495969798999A9B9C9D9E9F28\n"
		":10006000A0A1A2A3A4A5A6A7A8A9AAABACADAEAF18\n"
		":10007000B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBF08\n"
		":10008000C0C1C2C3C4C5C6C7C8C9CACBCCCDCECFF8\n"
		":10009000D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE8\n"
		":1000A000E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFD8\n"
		":1000B000F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFFC8\n"
		":00000001FF\n");
}

TEST(IntelHexExport, ErrorsIf16BitVirtAddressExceeds64K)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(0,        0, 64);
	doc->set_virt_mapping(64,  0xFFF0, 64);
	
	TempFilename tmp;
	
	EXPECT_THROW({
		try
		{
			write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_16BIT, NULL, NULL);
		}
		catch(const std::runtime_error &e)
		{
			EXPECT_STREQ("16-bit Intel Hex files cannot address beyond 64KiB", e.what());
			throw;
		}
	}, std::runtime_error);
}

TEST(IntelHexExport, ExportsFileWithExtendedSegmentAddresses)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(  0,      64, 64);
	doc->set_virt_mapping( 64,  0xFFFF, 64);
	doc->set_virt_mapping(128, 0x1007F, 64);
	doc->set_virt_mapping(192, 0x30001, 64);
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_SEGMENTED, NULL, NULL);
	
	EXPECT_FILE(
		/* 0x40 - 0xA4 */
		":10004000000102030405060708090A0B0C0D0E0F38\n"
		":10005000101112131415161718191A1B1C1D1E1F28\n"
		":10006000202122232425262728292A2B2C2D2E2F18\n"
		":10007000303132333435363738393A3B3C3D3E3F08\n"
		
		/* 0xFFFF - 0x1003F */
		":01FFFF0040C1\n"
		":020000021000EC\n"
		":100000004142434445464748494A4B4C4D4E4F5068\n"
		":100010005152535455565758595A5B5C5D5E5F6058\n"
		":100020006162636465666768696A6B6C6D6E6F7048\n"
		":0F0030007172737475767778797A7B7C7D7E7FB9\n"
		
		/* 0x1007F - 0x100BF */
		":10007F00808182838485868788898A8B8C8D8E8FF9\n"
		":10008F00909192939495969798999A9B9C9D9E9FE9\n"
		":10009F00A0A1A2A3A4A5A6A7A8A9AAABACADAEAFD9\n"
		":1000AF00B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC9\n"
		
		/* 0x30001 - 0x30041 */
		":020000023000CC\n"
		":10000100C0C1C2C3C4C5C6C7C8C9CACBCCCDCECF77\n"
		":10001100D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDF67\n"
		":10002100E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEF57\n"
		":10003100F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF47\n"
		
		":00000001FF\n");
}

TEST(IntelHexExport, ErrorsIfExtendedSegmentAddressExceeds1M)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(  0,      64, 64);
	doc->set_virt_mapping( 64,  0xFFFF, 64);
	doc->set_virt_mapping(128, 0x1007F, 64);
	doc->set_virt_mapping(192, 0xFFFFF, 64);
	
	TempFilename tmp;
	
	EXPECT_THROW({
		try
		{
			write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_SEGMENTED, NULL, NULL);
		}
		catch(const std::runtime_error &e)
		{
			EXPECT_STREQ("Segmented Intel Hex files cannot address beyond 1MiB", e.what());
			throw;
		}
	}, std::runtime_error);
}

TEST(IntelHexExport, ExportsFileWithLinearSegmentAddresses)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(  0,       0x40, 64);
	doc->set_virt_mapping( 64,    0xFFFFF, 64);
	doc->set_virt_mapping(128,   0x10007F, 64);
	doc->set_virt_mapping(192, 0x20000001, 64);
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_LINEAR, NULL, NULL);
	
	EXPECT_FILE(
		/* 0x40 - 0xA4 */
		":10004000000102030405060708090A0B0C0D0E0F38\n"
		":10005000101112131415161718191A1B1C1D1E1F28\n"
		":10006000202122232425262728292A2B2C2D2E2F18\n"
		":10007000303132333435363738393A3B3C3D3E3F08\n"
		
		/* 0xFFFFF - 0x10003F */
		":02000004000FEB\n"
		":01FFFF0040C1\n"
		":020000040010EA\n"
		":100000004142434445464748494A4B4C4D4E4F5068\n"
		":100010005152535455565758595A5B5C5D5E5F6058\n"
		":100020006162636465666768696A6B6C6D6E6F7048\n"
		":0F0030007172737475767778797A7B7C7D7E7FB9\n"
		
		/* 0x10007F - 0x1000BF */
		":10007F00808182838485868788898A8B8C8D8E8FF9\n"
		":10008F00909192939495969798999A9B9C9D9E9FE9\n"
		":10009F00A0A1A2A3A4A5A6A7A8A9AAABACADAEAFD9\n"
		":1000AF00B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC9\n"
		
		/* 0x20000001 - 0x20000041 */
		":020000042000DA\n"
		":10000100C0C1C2C3C4C5C6C7C8C9CACBCCCDCECF77\n"
		":10001100D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDF67\n"
		":10002100E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEF57\n"
		":10003100F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF47\n"
		
		":00000001FF\n");
}

TEST(IntelHexExport, ErrorsIfLinearSegmentAddressExceeds4G)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(  0,       0x40, 64);
	doc->set_virt_mapping( 64,     0xFFFF, 64);
	doc->set_virt_mapping(128,    0x1007F, 64);
	doc->set_virt_mapping(192, 0xFFFFFFC1, 64);
	
	TempFilename tmp;
	
	EXPECT_THROW({
		try
		{
			write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_LINEAR, NULL, NULL);
		}
		catch(const std::runtime_error &e)
		{
			EXPECT_STREQ("Linear Intel Hex files cannot address beyond 4GiB", e.what());
			throw;
		}
	}, std::runtime_error);
}

TEST(IntelHexExport, ExportsFileWithExtendedSegmentAddressesAndStartSegmentAddress)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(  0,      64, 64);
	doc->set_virt_mapping( 64,  0xFFFF, 64);
	doc->set_virt_mapping(128, 0x1007F, 64);
	doc->set_virt_mapping(192, 0x30001, 64);
	
	uint32_t start_segment_address = 0x12345678;
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_SEGMENTED, &start_segment_address, NULL);
	
	EXPECT_FILE(
		/* Start Segment Address */
		":0400000312345678E5\n"
		
		/* 0x40 - 0xA4 */
		":10004000000102030405060708090A0B0C0D0E0F38\n"
		":10005000101112131415161718191A1B1C1D1E1F28\n"
		":10006000202122232425262728292A2B2C2D2E2F18\n"
		":10007000303132333435363738393A3B3C3D3E3F08\n"
		
		/* 0xFFFF - 0x1003F */
		":01FFFF0040C1\n"
		":020000021000EC\n"
		":100000004142434445464748494A4B4C4D4E4F5068\n"
		":100010005152535455565758595A5B5C5D5E5F6058\n"
		":100020006162636465666768696A6B6C6D6E6F7048\n"
		":0F0030007172737475767778797A7B7C7D7E7FB9\n"
		
		/* 0x1007F - 0x100BF */
		":10007F00808182838485868788898A8B8C8D8E8FF9\n"
		":10008F00909192939495969798999A9B9C9D9E9FE9\n"
		":10009F00A0A1A2A3A4A5A6A7A8A9AAABACADAEAFD9\n"
		":1000AF00B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC9\n"
		
		/* 0x30001 - 0x30041 */
		":020000023000CC\n"
		":10000100C0C1C2C3C4C5C6C7C8C9CACBCCCDCECF77\n"
		":10001100D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDF67\n"
		":10002100E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEF57\n"
		":10003100F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF47\n"
		
		":00000001FF\n");
}

TEST(IntelHexExport, ExportsFileWithLinearSegmentAddressesAndStartLinearAddress)
{
	SharedDocumentPointer doc(SharedDocumentPointer::make());
	
	unsigned char data[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	
	doc->insert_data(0, data, sizeof(data));
	
	doc->set_virt_mapping(  0,       0x40, 64);
	doc->set_virt_mapping( 64,    0xFFFFF, 64);
	doc->set_virt_mapping(128,   0x10007F, 64);
	doc->set_virt_mapping(192, 0x20000001, 64);
	
	uint32_t start_linear_address = 0x87654321;
	
	TempFilename tmp;
	write_hex_file(tmp.tmpfile, doc, true, IntelHexAddressingMode::IHA_LINEAR, NULL, &start_linear_address);
	
	EXPECT_FILE(
		/* Start Linear Address */
		":0400000587654321A7\n"
		
		/* 0x40 - 0xA4 */
		":10004000000102030405060708090A0B0C0D0E0F38\n"
		":10005000101112131415161718191A1B1C1D1E1F28\n"
		":10006000202122232425262728292A2B2C2D2E2F18\n"
		":10007000303132333435363738393A3B3C3D3E3F08\n"
		
		/* 0xFFFFF - 0x10003F */
		":02000004000FEB\n"
		":01FFFF0040C1\n"
		":020000040010EA\n"
		":100000004142434445464748494A4B4C4D4E4F5068\n"
		":100010005152535455565758595A5B5C5D5E5F6058\n"
		":100020006162636465666768696A6B6C6D6E6F7048\n"
		":0F0030007172737475767778797A7B7C7D7E7FB9\n"
		
		/* 0x10007F - 0x1000BF */
		":10007F00808182838485868788898A8B8C8D8E8FF9\n"
		":10008F00909192939495969798999A9B9C9D9E9FE9\n"
		":10009F00A0A1A2A3A4A5A6A7A8A9AAABACADAEAFD9\n"
		":1000AF00B0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC9\n"
		
		/* 0x20000001 - 0x20000041 */
		":020000042000DA\n"
		":10000100C0C1C2C3C4C5C6C7C8C9CACBCCCDCECF77\n"
		":10001100D0D1D2D3D4D5D6D7D8D9DADBDCDDDEDF67\n"
		":10002100E0E1E2E3E4E5E6E7E8E9EAEBECEDEEEF57\n"
		":10003100F0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF47\n"
		
		":00000001FF\n");
}
