/* Reverse Engineer's Hex Editor
 * Copyright (C) 2022-2025 Daniel Collins <solemnwarning@solemnwarning.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "../src/platform.hpp"

#include <gtest/gtest.h>
#include <wx/event.h>
#include <wx/frame.h>

#include "testutil.hpp"
#include "../src/BitmapTool.hpp"
#include "../src/SharedDocumentPointer.hpp"

using namespace REHex;

class BitmapToolTest: public ::testing::Test
{
	protected:
		wxFrame frame;
		
		SharedDocumentPointer doc;
		DocumentCtrl *doc_ctrl;
		BitmapTool *bmtool;
		
		BitmapToolTest();
		
		static std::string bitmap_to_string(wxBitmap bitmap);
};

BitmapToolTest::BitmapToolTest():
	frame(NULL, wxID_ANY, "REHex Tests"),
	doc(SharedDocumentPointer::make())
{
	doc_ctrl = new DocumentCtrl(&frame, doc);
	
	/* Need a data region to avoid crashing in DocumentCtrl event handlers. */
	std::vector<DocumentCtrl::Region*> regions = { new DocumentCtrl::DataRegion(doc, 0, 0, 0) };
	doc_ctrl->replace_all_regions(regions);

	bmtool = new BitmapTool(&frame, doc, doc_ctrl);
}

std::string BitmapToolTest::bitmap_to_string(wxBitmap bitmap)
{
	wxImage image = bitmap.ConvertToImage();
	
	int width = image.GetWidth();
	int height = image.GetHeight();
	
	std::string s;
	s.reserve(width * height * 8);
	
	for(int y = 0; y < height; ++y)
	{
		for(int x = 0; x < width; ++x)
		{
			if(x > 0)
			{
				s += " ";
			}
			
			char colour[16];
			snprintf(colour, sizeof(colour), "#%02X%02X%02X",
				(unsigned int)(image.GetRed(x, y)),
				(unsigned int)(image.GetGreen(x, y)),
				(unsigned int)(image.GetBlue(x, y)));
			
			s += colour;
		}
		
		s += "\n";
	}
	
	return s;
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePacked)
{
	static const unsigned char PIXEL_DATA[] = {
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
	};
	
	static const char *EXPECT_PIXELS =
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePackedHalfScale)
{
	static const unsigned char PIXEL_DATA[] = {
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
	};
	
	static const char *EXPECT_PIXELS =
		"#FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000\n"
		"#000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF\n"
		"#FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000\n"
		"#000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF\n"
		"#FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000\n"
		"#000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF\n"
		"#FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000\n"
		"#000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF #000000 #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->force_bitmap_size(8, 8);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePackedDoubleScale)
{
	static const unsigned char PIXEL_DATA[] = {
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
	};
	
	static const char *EXPECT_PIXELS =
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->force_bitmap_size(32, 32);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePackedFlipX)
{
	static const unsigned char PIXEL_DATA[] = {
		0xFC, 0xCC,
		0xFC, 0xCC,
		0xF3, 0x33,
		0xF3, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
	};
	
	static const char *EXPECT_PIXELS =
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->set_flip_x(true);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePackedFlipY)
{
	static const unsigned char PIXEL_DATA[] = {
		0xFC, 0xCC,
		0xFC, 0xCC,
		0xF3, 0x33,
		0xF3, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
	};
	
	static const char *EXPECT_PIXELS =
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->set_flip_y(true);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePackedFlipXY)
{
	static const unsigned char PIXEL_DATA[] = {
		0xFC, 0xCC,
		0xFC, 0xCC,
		0xF3, 0x33,
		0xF3, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
		0xCC, 0xCC,
		0xCC, 0xCC,
		0x33, 0x33,
		0x33, 0x33,
	};
	
	static const char *EXPECT_PIXELS =
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->set_flip_x(true);
	bmtool->set_flip_y(true);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscalePadded)
{
	static const unsigned char PIXEL_DATA[] = {
		0xCC, 0xCC, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
		0x33, 0x33, 0xFF, 0xFF,
	};
	
	static const char *EXPECT_PIXELS =
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->set_row_length(4);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscaleOddPixelsPacked)
{
	static const unsigned char PIXEL_DATA[] = {
		// Pixel values
		// 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0
		// 1 0 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0 0
		// 0 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0 0 1
		// 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0 0 1 0
		// 1 1 0 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0
		// 1 0 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1
		// 0 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1 1
		// 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1 1 0
		// 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0
		// 0 1 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0
		// 1 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0 0
		// 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0 0 1
		// 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0 0 1 1
		// 1 0 0 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1
		// 0 0 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1 1
		// 0 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0
		// 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0 0
		// 0 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0 0 0
		
		                  // Pixel values (packed to 24 pixels/row)
		0x4C, 0x3C, 0x26, // 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0 1 0 0 1 1 0
		0x1E, 0x03, 0x0F, // 0 0 0 1 1 1 1 0 0 0 0 0 0 0 1 1 0 0 0 0 1 1 1 1
		0x05, 0x87, 0x82, // 0 0 0 0 0 1 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0 0 1 0
		0xC3, 0xC1, 0x21, // 1 1 0 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1 0 0 0 0 1
		0xE0, 0x90, 0xF0, // 1 1 1 0 0 0 0 0 1 0 0 1 0 0 0 0 1 1 1 1 0 0 0 0
		0x4C, 0x78, 0x26, // 0 1 0 0 1 1 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1 1 0
		0x3C, 0x13, 0x1E, // 0 0 1 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0 1 1 1 1 0
		0x09, 0x8F, 0x04, // 0 0 0 0 1 0 0 1 1 0 0 0 1 1 1 1 0 0 0 0 0 1 0 0
		0xC3, 0x82, 0x61, // 1 1 0 0 0 0 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0 0 1
		0xC1, 0x30, 0xE0, // 1 1 0 0 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1 0 0 0 0 0
		0x98, 0x70, 0x4C, // 1 0 0 1 1 0 0 0 0 1 1 1 0 0 0 0 0 1 0 0 1 1 0 0
		0x3C, 0x26, 0x1E, // 0 0 1 1 1 1 0 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0
		0x13, 0x0F, 0x09, // 0 0 0 1 0 0 1 1 0 0 0 0 1 1 1 1 0 0 0 0 1 0 0 1
		0x87, 0x8F, 0xFF, // 1 0 0 0 0 1 1 1 1 0 0 0
	};
	
	static const char *EXPECT_PIXELS =
		"#000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF\n"
		"#000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000\n"
		"#FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000\n"
		"#FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF\n"
		"#000000 #000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000\n"
		"#000000 #000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #000000 #000000 #FFFFFF #FFFFFF #FFFFFF #FFFFFF #000000 #000000 #000000\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(18, 18);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->force_bitmap_size(18, 18);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format1BPPGreyscaleOddPixelsPadded)
{
	static const unsigned char PIXEL_DATA[] = {
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0x33, 0x33, 0x3F, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
		0xCC, 0xCC, 0xFF, 0xFF, 0xFF,
	};
	
	static const char *EXPECT_PIXELS =
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n"
		"#FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF #000000 #000000 #FFFFFF #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(18, 18);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_1BPP);
	bmtool->set_row_length(5);
	bmtool->force_bitmap_size(18, 18);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format2BPPGreyscalePacked)
{
	static const unsigned char PIXEL_DATA[] = {
		0x05, 0xAF, 0x05, 0xAF, // 0 0 1 1 2 2 3 3 0 0 1 1 2 2 3 3
		0x5A, 0xF0, 0x5A, 0xF0, // 1 1 2 2 3 3 0 0 1 1 2 2 3 3 0 0
		0xAF, 0x05, 0xAF, 0x05, // 2 2 3 3 0 0 1 1 2 2 3 3 0 0 1 1
		0xF0, 0x5A, 0xF0, 0x5A, // 3 3 0 0 1 1 2 2 3 3 0 0 1 1 2 2
		0x05, 0xAF, 0x05, 0xAF, // 0 0 1 1 2 2 3 3 0 0 1 1 2 2 3 3
		0x5A, 0xF0, 0x5A, 0xF0, // 1 1 2 2 3 3 0 0 1 1 2 2 3 3 0 0
		0xAF, 0x05, 0xAF, 0x05, // 2 2 3 3 0 0 1 1 2 2 3 3 0 0 1 1
		0xF0, 0x5A, 0xF0, 0x5A, // 3 3 0 0 1 1 2 2 3 3 0 0 1 1 2 2
		0x05, 0xAF, 0x05, 0xAF, // 0 0 1 1 2 2 3 3 0 0 1 1 2 2 3 3
		0x5A, 0xF0, 0x5A, 0xF0, // 1 1 2 2 3 3 0 0 1 1 2 2 3 3 0 0
		0xAF, 0x05, 0xAF, 0x05, // 2 2 3 3 0 0 1 1 2 2 3 3 0 0 1 1
		0xF0, 0x5A, 0xF0, 0x5A, // 3 3 0 0 1 1 2 2 3 3 0 0 1 1 2 2
		0x05, 0xAF, 0x05, 0xAF, // 0 0 1 1 2 2 3 3 0 0 1 1 2 2 3 3
		0x5A, 0xF0, 0x5A, 0xF0, // 1 1 2 2 3 3 0 0 1 1 2 2 3 3 0 0
		0xAF, 0x05, 0xAF, 0x05, // 2 2 3 3 0 0 1 1 2 2 3 3 0 0 1 1
		0xF0, 0x5A, 0xF0, 0x5A, // 3 3 0 0 1 1 2 2 3 3 0 0 1 1 2 2
	};
	
	static const char *EXPECT_PIXELS =
		"#000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF\n"
		"#555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000\n"
		"#AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555\n"
		"#FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA\n"
		"#000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF\n"
		"#555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000\n"
		"#AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555\n"
		"#FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA\n"
		"#000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF\n"
		"#555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000\n"
		"#AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555\n"
		"#FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA\n"
		"#000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF\n"
		"#555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000\n"
		"#AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555\n"
		"#FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA #FFFFFF #FFFFFF #000000 #000000 #555555 #555555 #AAAAAA #AAAAAA\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_2BPP);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format4BPPGreyscalePacked)
{
	static const unsigned char PIXEL_DATA[] = {
		0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF, // 0 1 2 3 4 5 6 7 8 9 A B C D E F
		0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0, // 1 2 3 4 5 6 7 8 9 A B C D E F 0
		0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF, 0x01, // 2 3 4 5 6 7 8 9 A B C D E F 0 1
		0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0, 0x12, // 3 4 5 6 7 8 9 A B C D E F 0 1 2
		0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF, 0x01, 0x23, // 4 5 6 7 8 9 A B C D E F 0 1 2 3
		0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0, 0x12, 0x34, // 5 6 7 8 9 A B C D E F 0 1 2 3 4
		0x67, 0x89, 0xAB, 0xCD, 0xEF, 0x01, 0x23, 0x45, // 6 7 8 9 A B C D E F 0 1 2 3 4 5
		0x78, 0x9A, 0xBC, 0xDE, 0xF0, 0x12, 0x34, 0x56, // 7 8 9 A B C D E F 0 1 2 3 4 5 6
		0x89, 0xAB, 0xCD, 0xEF, 0x01, 0x23, 0x45, 0x67, // 8 9 A B C D E F 0 1 2 3 4 5 6 7
		0x9A, 0xBC, 0xDE, 0xF0, 0x12, 0x34, 0x56, 0x78, // 9 A B C D E F 0 1 2 3 4 5 6 7 8
		0xAB, 0xCD, 0xEF, 0x01, 0x23, 0x45, 0x67, 0x89, // A B C D E F 0 1 2 3 4 5 6 7 8 9
		0xBC, 0xDE, 0xF0, 0x12, 0x34, 0x56, 0x78, 0x9A, // B C D E F 0 1 2 3 4 5 6 7 8 9 A
		0xCD, 0xEF, 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, // C D E F 0 1 2 3 4 5 6 7 8 9 A B
		0xDE, 0xF0, 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, // D E F 0 1 2 3 4 5 6 7 8 9 A B C
		0xEF, 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, // E F 0 1 2 3 4 5 6 7 8 9 A B C D
		0xF0, 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, // F 0 1 2 3 4 5 6 7 8 9 A B C D E
	};
	
	static const char *EXPECT_PIXELS =
		"#000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF\n"
		"#111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000\n"
		"#222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111\n"
		"#333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222\n"
		"#444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333\n"
		"#555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444\n"
		"#666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555\n"
		"#777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666\n"
		"#888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777\n"
		"#999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888\n"
		"#AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999\n"
		"#BBBBBB #CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA\n"
		"#CCCCCC #DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB\n"
		"#DDDDDD #EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC\n"
		"#EEEEEE #FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD\n"
		"#FFFFFF #000000 #111111 #222222 #333333 #444444 #555555 #666666 #777777 #888888 #999999 #AAAAAA #BBBBBB #CCCCCC #DDDDDD #EEEEEE\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_4BPP);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format8BPPGreyscalePacked)
{
	static const unsigned char PIXEL_DATA[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F,
		0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F,
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F,
		0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,
		0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF,
		0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF,
		0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,
		0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
		0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
	};
	
	static const char *EXPECT_PIXELS =
		"#000000 #010101 #020202 #030303 #040404 #050505 #060606 #070707 #080808 #090909 #0A0A0A #0B0B0B #0C0C0C #0D0D0D #0E0E0E #0F0F0F\n"
		"#101010 #111111 #121212 #131313 #141414 #151515 #161616 #171717 #181818 #191919 #1A1A1A #1B1B1B #1C1C1C #1D1D1D #1E1E1E #1F1F1F\n"
		"#202020 #212121 #222222 #232323 #242424 #252525 #262626 #272727 #282828 #292929 #2A2A2A #2B2B2B #2C2C2C #2D2D2D #2E2E2E #2F2F2F\n"
		"#303030 #313131 #323232 #333333 #343434 #353535 #363636 #373737 #383838 #393939 #3A3A3A #3B3B3B #3C3C3C #3D3D3D #3E3E3E #3F3F3F\n"
		"#404040 #414141 #424242 #434343 #444444 #454545 #464646 #474747 #484848 #494949 #4A4A4A #4B4B4B #4C4C4C #4D4D4D #4E4E4E #4F4F4F\n"
		"#505050 #515151 #525252 #535353 #545454 #555555 #565656 #575757 #585858 #595959 #5A5A5A #5B5B5B #5C5C5C #5D5D5D #5E5E5E #5F5F5F\n"
		"#606060 #616161 #626262 #636363 #646464 #656565 #666666 #676767 #686868 #696969 #6A6A6A #6B6B6B #6C6C6C #6D6D6D #6E6E6E #6F6F6F\n"
		"#707070 #717171 #727272 #737373 #747474 #757575 #767676 #777777 #787878 #797979 #7A7A7A #7B7B7B #7C7C7C #7D7D7D #7E7E7E #7F7F7F\n"
		"#808080 #818181 #828282 #838383 #848484 #858585 #868686 #878787 #888888 #898989 #8A8A8A #8B8B8B #8C8C8C #8D8D8D #8E8E8E #8F8F8F\n"
		"#909090 #919191 #929292 #939393 #949494 #959595 #969696 #979797 #989898 #999999 #9A9A9A #9B9B9B #9C9C9C #9D9D9D #9E9E9E #9F9F9F\n"
		"#A0A0A0 #A1A1A1 #A2A2A2 #A3A3A3 #A4A4A4 #A5A5A5 #A6A6A6 #A7A7A7 #A8A8A8 #A9A9A9 #AAAAAA #ABABAB #ACACAC #ADADAD #AEAEAE #AFAFAF\n"
		"#B0B0B0 #B1B1B1 #B2B2B2 #B3B3B3 #B4B4B4 #B5B5B5 #B6B6B6 #B7B7B7 #B8B8B8 #B9B9B9 #BABABA #BBBBBB #BCBCBC #BDBDBD #BEBEBE #BFBFBF\n"
		"#C0C0C0 #C1C1C1 #C2C2C2 #C3C3C3 #C4C4C4 #C5C5C5 #C6C6C6 #C7C7C7 #C8C8C8 #C9C9C9 #CACACA #CBCBCB #CCCCCC #CDCDCD #CECECE #CFCFCF\n"
		"#D0D0D0 #D1D1D1 #D2D2D2 #D3D3D3 #D4D4D4 #D5D5D5 #D6D6D6 #D7D7D7 #D8D8D8 #D9D9D9 #DADADA #DBDBDB #DCDCDC #DDDDDD #DEDEDE #DFDFDF\n"
		"#E0E0E0 #E1E1E1 #E2E2E2 #E3E3E3 #E4E4E4 #E5E5E5 #E6E6E6 #E7E7E7 #E8E8E8 #E9E9E9 #EAEAEA #EBEBEB #ECECEC #EDEDED #EEEEEE #EFEFEF\n"
		"#F0F0F0 #F1F1F1 #F2F2F2 #F3F3F3 #F4F4F4 #F5F5F5 #F6F6F6 #F7F7F7 #F8F8F8 #F9F9F9 #FAFAFA #FBFBFB #FCFCFC #FDFDFD #FEFEFE #FFFFFF\n";
	
	doc->insert_data(0, PIXEL_DATA, sizeof(PIXEL_DATA));
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(16, 16);
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_8BPP_GREYSCALE);
	bmtool->force_bitmap_size(16, 16);
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	EXPECT_EQ(bitmap_pixels, EXPECT_PIXELS);
}

TEST_F(BitmapToolTest, Format24BPPRGBPackedBigImage)
{
	wxImage ref_img("res/icon1024.png", wxBITMAP_TYPE_ANY);
	if(ref_img.HasAlpha())
	{
		ref_img.ClearAlpha();
	}
	
	/* Clean up the image to eliminate noise from stripping the alpha
	 * channel (only required for older wxWidgets versions).
	*/
	ref_img = wxBitmap(ref_img).ConvertToImage();
	
	int ref_img_w = ref_img.GetWidth();
	ASSERT_EQ(ref_img_w, 1024);
	
	int ref_img_h = ref_img.GetHeight();
	ASSERT_EQ(ref_img_h, 1024);
	
	std::string EXPECT_PIXELS = bitmap_to_string(wxBitmap(ref_img));
	
	doc->insert_data(0, ref_img.GetData(), ref_img_w * ref_img_h * 3);
	
	bmtool->set_image_offset(0);
	bmtool->set_image_size(ref_img.GetWidth(), ref_img.GetHeight());
	bmtool->set_pixel_format(BitmapTool::PIXEL_FMT_24BPP_RGB888);
	bmtool->force_bitmap_size(ref_img.GetWidth(), ref_img.GetHeight());
	
	run_wx_until([&]() { return !bmtool->is_processing(); });
	
	std::string bitmap_pixels = bitmap_to_string(bmtool->get_bitmap());
	
	/* Too huge to display meaningful failures in the test log. */
	EXPECT_TRUE(bitmap_pixels == EXPECT_PIXELS);
}
