/* Reverse Engineer's Hex Editor
 * Copyright (C) 2017-2024 Daniel Collins <solemnwarning@solemnwarning.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "BufferTest.h"

TEST(Buffer, EraseTinyFileStart)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x77, 0xA4, 0xE2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 3);
			});
			
			TEST_LENGTH(3);
		}
	);
}

TEST(Buffer, EraseTinyFileEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0xF8, 0xD1,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(2, 3);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 2);
			});
			
			TEST_LENGTH(2);
		}
	);
}

TEST(Buffer, EraseTinyFileMiddle)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0xF8, 0xD1, 0xE2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(2, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 3);
			});
			
			TEST_LENGTH(3);
		}
	);
}

TEST(Buffer, EraseTinyFileAll)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	const std::vector<unsigned char> END_DATA = {};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 5);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 0);
			});
			
			TEST_LENGTH(0);
		}
	);
}

TEST(Buffer, EraseTinyFileAllAndThemSome)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(0, 6);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0, 5);
			});
			
			TEST_LENGTH(5);
		}
	);
}

TEST(Buffer, EraseTinyFileFromEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0xF8, 0xD1, 0x77, 0xA4, 0xE2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(5, 1);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0, 5);
			});
			
			TEST_LENGTH(5);
		}
	);
}

TEST(Buffer, EraseSingleBlockFileStart)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		/* 0xF0, 0x0D, */ 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 6);
			});
			
			TEST_LENGTH(6);
		}
	);
}

TEST(Buffer, EraseSingleBlockFileEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, /* 0x74, 0x50, 0xD2, */
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(5, 3);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 5);
			});
			
			TEST_LENGTH(5);
		}
	);
}

TEST(Buffer, EraseSingleBlockFileMiddle)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x09, 0x7E, /* 0x9B, 0x25, 0xCB, 0x74, */ 0x50, 0xD2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(2, 4);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 4);
			});
			
			TEST_LENGTH(4);
		}
	);
}

TEST(Buffer, EraseSingleBlockFileAll)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		/* 0x34, 0x89, 0x3D, 0x7B, 0x6F, 0xBF, 0x13, 0xC0, */
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 8);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 0);
			});
			
			TEST_LENGTH(0);
		}
	);
}

TEST(Buffer, EraseSingleBlockFileAllAndThenSome)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(0, 9);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0, 8);
			});
			
			TEST_LENGTH(8);
		}
	);
}

TEST(Buffer, EraseSingleBlockFileFromEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x09, 0x7E, 0x9B, 0x25, 0xCB, 0x74, 0x50, 0xD2,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(8, 1);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0, 8);
			});
			
			TEST_LENGTH(8);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileFirstBlockStart)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		/* 0x06, 0x96, */ 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0,  6);
				TEST_BLOCK_DEF(UNLOADED, 6,  8);
				TEST_BLOCK_DEF(UNLOADED, 14, 8);
				TEST_BLOCK_DEF(UNLOADED, 22, 6);
			});
			
			TEST_LENGTH(28);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileFirstBlockEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, /* 0xB5, 0x99, 0x4E, */
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(5, 3);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0,  5);
				TEST_BLOCK_DEF(UNLOADED, 5,  8);
				TEST_BLOCK_DEF(UNLOADED, 13, 8);
				TEST_BLOCK_DEF(UNLOADED, 21, 6);
			});
			
			TEST_LENGTH(27);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileSecondBlockStart)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		/* 0xE7, 0xA8, 0x06, */ 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(8, 3);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(DIRTY,    8,  5);
				TEST_BLOCK_DEF(UNLOADED, 13, 8);
				TEST_BLOCK_DEF(UNLOADED, 21, 6);
			});
			
			TEST_LENGTH(27);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileSecondBlockEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, /* 0x8C, 0xD1, */
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(14, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(DIRTY,    8,  6);
				TEST_BLOCK_DEF(UNLOADED, 14, 8);
				TEST_BLOCK_DEF(UNLOADED, 22, 6);
			});
			
			TEST_LENGTH(28);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileLastBlockStart)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		/* 0x51, 0xA0, */ 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(24, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(UNLOADED, 8,  8);
				TEST_BLOCK_DEF(UNLOADED, 16, 8);
				TEST_BLOCK_DEF(DIRTY,    24, 4);
			});
			
			TEST_LENGTH(28);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileLastBlockEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, /* 0x67, 0xC9, */
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(28, 2);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(UNLOADED, 8,  8);
				TEST_BLOCK_DEF(UNLOADED, 16, 8);
				TEST_BLOCK_DEF(DIRTY,    24, 4);
			});
			
			TEST_LENGTH(28);
		}
	);
}

TEST(Buffer, EraseMultiBlockFilePartialBlocks)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, /* 0x99, 0x4E,
		0xE7, 0xA8, */ 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(6, 4);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0,  6);
				TEST_BLOCK_DEF(DIRTY,    6,  6);
				TEST_BLOCK_DEF(UNLOADED, 12, 8);
				TEST_BLOCK_DEF(UNLOADED, 20, 6);
			});
			
			TEST_LENGTH(26);
		}
	);
}

TEST(Buffer, EraseMultiBlockFilePartialBlocksMore)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, /* 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, */ 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(3, 24);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 3);
				TEST_BLOCK_DEF(DIRTY, 3, 0);
				TEST_BLOCK_DEF(DIRTY, 3, 0);
				TEST_BLOCK_DEF(DIRTY, 3, 3);
			});
			
			TEST_LENGTH(6);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileAll)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 30);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY, 0, 0);
				TEST_BLOCK_DEF(DIRTY, 0, 0);
				TEST_BLOCK_DEF(DIRTY, 0, 0);
				TEST_BLOCK_DEF(DIRTY, 0, 0);
			});
			
			TEST_LENGTH(0);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileAllAndThemSome)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(0, 31);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(UNLOADED, 8,  8);
				TEST_BLOCK_DEF(UNLOADED, 16, 8);
				TEST_BLOCK_DEF(UNLOADED, 24, 6);
			});
			
			TEST_LENGTH(30);
		}
	);
}

TEST(Buffer, EraseMultiBlockFileFromEnd)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(30, 1);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(UNLOADED, 8,  8);
				TEST_BLOCK_DEF(UNLOADED, 16, 8);
				TEST_BLOCK_DEF(UNLOADED, 24, 6);
			});
			
			TEST_LENGTH(30);
		}
	);
}

TEST(Buffer, EraseEmptyFileStart)
{
	const std::vector<unsigned char> BEGIN_DATA = {};
	const std::vector<unsigned char> END_DATA   = {};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_FAIL(0, 1);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0, 0);
			});
			
			TEST_LENGTH(0);
		}
	);
}

TEST(Buffer, EraseZeroBytes)
{
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
	};
	
	const std::vector<unsigned char> END_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 0);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(UNLOADED, 0, 8);
			});
			
			TEST_LENGTH(8);
		}
	);
}

TEST(Buffer, EraseSequence1)
{
	/* Test erasing in sequence so we can see erase_data() handles
	 * zero-length blocks and blocks with the same offset correctly.
	*/
	
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		/* 0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E, */
		/* 0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1, */
		/* 0xE0, 0x3B, 0x0F, 0x7C, */ 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 8);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0,  0);
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(UNLOADED, 8,  8);
				TEST_BLOCK_DEF(UNLOADED, 16, 6);
			});
			
			TEST_LENGTH(22);
			
			TEST_ERASE_OK(0, 8);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(UNLOADED, 0, 8);
				TEST_BLOCK_DEF(UNLOADED, 8, 6);
			});
			
			TEST_LENGTH(14);
			
			TEST_ERASE_OK(0, 4);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(DIRTY,    0, 4);
				TEST_BLOCK_DEF(UNLOADED, 4, 6);
			});
			
			TEST_LENGTH(10);
		}
	);
}

TEST(Buffer, EraseSequence2)
{
	/* Test erasing in sequence so we can see erase_data() handles
	 * zero-length blocks and blocks with the same offset correctly.
	*/
	
	const std::vector<unsigned char> BEGIN_DATA = {
		0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E,
		0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1,
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		0x51, 0xA0, 0x0D, 0xAD, 0x67, 0xC9,
	};
	
	const std::vector<unsigned char> END_DATA = {
		/* 0x06, 0x96, 0x64, 0x58, 0xC9, 0xB5, 0x99, 0x4E, */
		/* 0xE7, 0xA8, 0x06, 0x24, 0xEC, 0xB6, 0x8C, 0xD1, */
		0xE0, 0x3B, 0x0F, 0x7C, 0xAD, 0x80, 0xB3, 0xB4,
		/* 0x51, 0xA0, 0x0D, 0xAD, */ 0x67, 0xC9,
	};
	
	TEST_BUFFER_MANIP(
		{
			TEST_ERASE_OK(0, 8);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0,  0);
				TEST_BLOCK_DEF(UNLOADED, 0,  8);
				TEST_BLOCK_DEF(UNLOADED, 8,  8);
				TEST_BLOCK_DEF(UNLOADED, 16, 6);
			});
			
			TEST_LENGTH(22);
			
			TEST_ERASE_OK(0, 8);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(UNLOADED, 0, 8);
				TEST_BLOCK_DEF(UNLOADED, 8, 6);
			});
			
			TEST_LENGTH(14);
			
			TEST_ERASE_OK(8, 4);
			
			TEST_BLOCKS({
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(DIRTY,    0, 0);
				TEST_BLOCK_DEF(UNLOADED, 0, 8);
				TEST_BLOCK_DEF(DIRTY,    8, 2);
			});
			
			TEST_LENGTH(10);
		}
	);
}
